---
name: instantly-sdk-patterns
description: |
  Apply production-ready Instantly SDK patterns for TypeScript and Python.
  Use when implementing Instantly integrations, refactoring SDK usage,
  or establishing team coding standards for Instantly.
  Trigger with phrases like "instantly SDK patterns", "instantly best practices",
  "instantly code patterns", "idiomatic instantly".
allowed-tools: Read, Write, Edit
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# Instantly SDK Patterns

## Overview
Production-ready patterns for Instantly SDK usage in TypeScript and Python.

## Prerequisites
- Completed `instantly-install-auth` setup
- Familiarity with async/await patterns
- Understanding of error handling best practices

## Instructions

### Step 1: Implement Singleton Pattern (Recommended)
```typescript
// src/instantly/client.ts
import { InstantlyClient } from '@instantly/sdk';

let instance: InstantlyClient | null = null;

export function getInstantlyClient(): InstantlyClient {
  if (!instance) {
    instance = new InstantlyClient({
      apiKey: process.env.INSTANTLY_API_KEY!,
      // Additional options
    });
  }
  return instance;
}
```

### Step 2: Add Error Handling Wrapper
```typescript
import { InstantlyError } from '@instantly/sdk';

async function safeInstantlyCall<T>(
  operation: () => Promise<T>
): Promise<{ data: T | null; error: Error | null }> {
  try {
    const data = await operation();
    return { data, error: null };
  } catch (err) {
    if (err instanceof InstantlyError) {
      console.error({
        code: err.code,
        message: err.message,
      });
    }
    return { data: null, error: err as Error };
  }
}
```

### Step 3: Implement Retry Logic
```typescript
async function withRetry<T>(
  operation: () => Promise<T>,
  maxRetries = 3,
  backoffMs = 1000
): Promise<T> {
  for (let attempt = 1; attempt <= maxRetries; attempt++) {
    try {
      return await operation();
    } catch (err) {
      if (attempt === maxRetries) throw err;
      const delay = backoffMs * Math.pow(2, attempt - 1);
      await new Promise(r => setTimeout(r, delay));
    }
  }
  throw new Error('Unreachable');
}
```

## Output
- Type-safe client singleton
- Robust error handling with structured logging
- Automatic retry with exponential backoff
- Runtime validation for API responses

## Error Handling
| Pattern | Use Case | Benefit |
|---------|----------|---------|
| Safe wrapper | All API calls | Prevents uncaught exceptions |
| Retry logic | Transient failures | Improves reliability |
| Type guards | Response validation | Catches API changes |
| Logging | All operations | Debugging and monitoring |

## Examples

### Factory Pattern (Multi-tenant)
```typescript
const clients = new Map<string, InstantlyClient>();

export function getClientForTenant(tenantId: string): InstantlyClient {
  if (!clients.has(tenantId)) {
    const apiKey = getTenantApiKey(tenantId);
    clients.set(tenantId, new InstantlyClient({ apiKey }));
  }
  return clients.get(tenantId)!;
}
```

### Python Context Manager
```python
from contextlib import asynccontextmanager
from instantly import InstantlyClient

@asynccontextmanager
async def get_instantly_client():
    client = InstantlyClient()
    try:
        yield client
    finally:
        await client.close()
```

### Zod Validation
```typescript
import { z } from 'zod';

const instantlyResponseSchema = z.object({
  id: z.string(),
  status: z.enum(['active', 'inactive']),
  createdAt: z.string().datetime(),
});
```

## Resources
- [Instantly SDK Reference](https://docs.instantly.com/sdk)
- [Instantly API Types](https://docs.instantly.com/types)
- [Zod Documentation](https://zod.dev/)

## Next Steps
Apply patterns in `instantly-core-workflow-a` for real-world usage.