---
name: openevidence-sdk-patterns
description: |
  OpenEvidence SDK patterns and best practices for clinical AI integration.
  Use when implementing advanced SDK features, optimizing API usage,
  or following clinical decision support best practices.
  Trigger with phrases like "openevidence patterns", "openevidence best practices",
  "openevidence sdk", "clinical ai patterns".
allowed-tools: Read, Write, Edit
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# OpenEvidence SDK Patterns

## Overview
Best practices and design patterns for building production clinical decision support with OpenEvidence.

## Prerequisites
- Completed `openevidence-install-auth` setup
- Understanding of async/await patterns
- Familiarity with clinical workflows

## Core Patterns

### Pattern 1: Client Singleton with Dependency Injection
```typescript
// src/openevidence/client-factory.ts
import { OpenEvidenceClient } from '@openevidence/sdk';

export interface OpenEvidenceConfig {
  apiKey: string;
  orgId: string;
  baseUrl?: string;
  timeout?: number;
  retries?: number;
}

export class OpenEvidenceClientFactory {
  private static instance: OpenEvidenceClient;
  private static config: OpenEvidenceConfig;

  static configure(config: OpenEvidenceConfig): void {
    this.config = config;
    this.instance = null!; // Reset on reconfigure
  }

  static getClient(): OpenEvidenceClient {
    if (!this.instance) {
      if (!this.config) {
        throw new Error('OpenEvidence client not configured. Call configure() first.');
      }
      this.instance = new OpenEvidenceClient(this.config);
    }
    return this.instance;
  }

  // For testing
  static setClient(client: OpenEvidenceClient): void {
    this.instance = client;
  }
}
```

### Pattern 2: Typed Clinical Queries
```typescript
// src/openevidence/types.ts
export type ClinicalSpecialty =
  | 'internal-medicine'
  | 'emergency-medicine'
  | 'cardiology'
  | 'oncology'
  | 'neurology'
  | 'pediatrics'
  | 'psychiatry'
  | 'surgery'
  | 'family-medicine'
  | 'pharmacology';

export type QueryUrgency = 'stat' | 'urgent' | 'routine' | 'research';

export interface ClinicalContext {
  specialty: ClinicalSpecialty;
  urgency: QueryUrgency;
  patientAge?: number;
  patientSex?: 'male' | 'female' | 'other';
  relevantConditions?: string[];
  currentMedications?: string[];
}

export interface QueryOptions {
  maxCitations?: number;
  includeGuidelines?: boolean;
  includeDrugInfo?: boolean;
  preferredSources?: string[];
}

export interface TypedClinicalQuery {
  question: string;
  context: ClinicalContext;
  options?: QueryOptions;
}
```

### Pattern 3: Query Builder Pattern
```typescript
// src/openevidence/query-builder.ts
import { TypedClinicalQuery, ClinicalContext, QueryOptions } from './types';

export class ClinicalQueryBuilder {
  private query: Partial<TypedClinicalQuery> = {};

  question(q: string): this {
    this.query.question = q;
    return this;
  }

  specialty(s: ClinicalContext['specialty']): this {
    this.query.context = { ...this.query.context, specialty: s } as ClinicalContext;
    return this;
  }

  urgency(u: ClinicalContext['urgency']): this {
    this.query.context = { ...this.query.context, urgency: u } as ClinicalContext;
    return this;
  }

  withPatient(age: number, sex: 'male' | 'female' | 'other'): this {
    this.query.context = {
      ...this.query.context,
      patientAge: age,
      patientSex: sex,
    } as ClinicalContext;
    return this;
  }

  withConditions(conditions: string[]): this {
    this.query.context = {
      ...this.query.context,
      relevantConditions: conditions,
    } as ClinicalContext;
    return this;
  }

  withMedications(meds: string[]): this {
    this.query.context = {
      ...this.query.context,
      currentMedications: meds,
    } as ClinicalContext;
    return this;
  }

  maxCitations(n: number): this {
    this.query.options = { ...this.query.options, maxCitations: n };
    return this;
  }

  includeGuidelines(): this {
    this.query.options = { ...this.query.options, includeGuidelines: true };
    return this;
  }

  build(): TypedClinicalQuery {
    if (!this.query.question) throw new Error('Question is required');
    if (!this.query.context?.specialty) throw new Error('Specialty is required');
    if (!this.query.context?.urgency) throw new Error('Urgency is required');

    return this.query as TypedClinicalQuery;
  }
}

// Usage
const query = new ClinicalQueryBuilder()
  .question('What is the recommended statin dosing for secondary prevention?')
  .specialty('cardiology')
  .urgency('routine')
  .withPatient(65, 'male')
  .withConditions(['prior MI', 'hyperlipidemia'])
  .includeGuidelines()
  .maxCitations(5)
  .build();
```

### Pattern 4: Response Transformer
```typescript
// src/openevidence/response-transformer.ts
interface RawOpenEvidenceResponse {
  answer: string;
  citations: any[];
  confidence: number;
  raw_data: any;
}

export interface FormattedClinicalAnswer {
  summary: string;
  detailedAnswer: string;
  keyPoints: string[];
  evidence: {
    source: string;
    strength: 'high' | 'moderate' | 'low';
    year: number;
  }[];
  confidence: {
    score: number;
    level: 'high' | 'moderate' | 'low';
  };
  disclaimer: string;
}

export function transformResponse(raw: RawOpenEvidenceResponse): FormattedClinicalAnswer {
  const keyPoints = extractKeyPoints(raw.answer);

  return {
    summary: raw.answer.split('.')[0] + '.',
    detailedAnswer: raw.answer,
    keyPoints,
    evidence: raw.citations.map(c => ({
      source: c.source,
      strength: determineEvidenceStrength(c),
      year: c.year,
    })),
    confidence: {
      score: raw.confidence,
      level: raw.confidence > 0.9 ? 'high' : raw.confidence > 0.7 ? 'moderate' : 'low',
    },
    disclaimer: 'Clinical answers are for informational purposes. Always verify with current guidelines.',
  };
}

function extractKeyPoints(answer: string): string[] {
  // Extract bullet points or numbered items from the answer
  const lines = answer.split('\n');
  return lines
    .filter(l => /^[-•*\d]/.test(l.trim()))
    .map(l => l.replace(/^[-•*\d.)\]]+\s*/, '').trim())
    .filter(l => l.length > 0);
}

function determineEvidenceStrength(citation: any): 'high' | 'moderate' | 'low' {
  const highImpactSources = ['NEJM', 'JAMA', 'Lancet', 'BMJ', 'NCCN'];
  if (highImpactSources.some(s => citation.source?.includes(s))) return 'high';
  if (citation.year >= new Date().getFullYear() - 2) return 'moderate';
  return 'low';
}
```

### Pattern 5: Caching Strategy
```typescript
// src/openevidence/cache.ts
import { createHash } from 'crypto';

interface CacheEntry<T> {
  data: T;
  timestamp: number;
  ttl: number;
}

export class ClinicalQueryCache {
  private cache = new Map<string, CacheEntry<any>>();
  private defaultTTL = 3600000; // 1 hour for clinical data

  private generateKey(query: any): string {
    const normalized = JSON.stringify(query, Object.keys(query).sort());
    return createHash('sha256').update(normalized).digest('hex');
  }

  get<T>(query: any): T | null {
    const key = this.generateKey(query);
    const entry = this.cache.get(key);

    if (!entry) return null;

    if (Date.now() - entry.timestamp > entry.ttl) {
      this.cache.delete(key);
      return null;
    }

    return entry.data as T;
  }

  set<T>(query: any, data: T, ttl?: number): void {
    const key = this.generateKey(query);
    this.cache.set(key, {
      data,
      timestamp: Date.now(),
      ttl: ttl || this.defaultTTL,
    });
  }

  invalidate(query: any): void {
    const key = this.generateKey(query);
    this.cache.delete(key);
  }

  clear(): void {
    this.cache.clear();
  }
}
```

## Output
- Typed client factory with DI support
- Query builder for complex clinical queries
- Response transformer for consistent output
- Caching layer for performance

## Error Handling
| Pattern | Use Case | Benefit |
|---------|----------|---------|
| Singleton | Single client instance | Memory efficiency |
| Builder | Complex queries | Type safety, readability |
| Transformer | Response normalization | Consistent UI layer |
| Cache | Repeated queries | Reduced API calls, latency |

## Examples

### Complete Service Implementation
```typescript
// src/services/clinical-evidence.ts
import { OpenEvidenceClientFactory } from '../openevidence/client-factory';
import { ClinicalQueryBuilder } from '../openevidence/query-builder';
import { transformResponse, FormattedClinicalAnswer } from '../openevidence/response-transformer';
import { ClinicalQueryCache } from '../openevidence/cache';

const cache = new ClinicalQueryCache();

export async function getClinicalEvidence(
  question: string,
  specialty: string,
  options?: { useCache?: boolean }
): Promise<FormattedClinicalAnswer> {
  const query = new ClinicalQueryBuilder()
    .question(question)
    .specialty(specialty as any)
    .urgency('routine')
    .includeGuidelines()
    .build();

  // Check cache first
  if (options?.useCache !== false) {
    const cached = cache.get<FormattedClinicalAnswer>(query);
    if (cached) return cached;
  }

  // Query API
  const client = OpenEvidenceClientFactory.getClient();
  const response = await client.query(query);

  // Transform and cache
  const formatted = transformResponse(response);
  cache.set(query, formatted);

  return formatted;
}
```

## Resources
- [OpenEvidence API Reference](https://docs.openevidence.com/)
- [TypeScript Design Patterns](https://refactoring.guru/design-patterns/typescript)

## Next Steps
For core clinical query workflow, see `openevidence-core-workflow-a`.
