/**
 * Format issue body with markdown
 */
export function formatIssueBody(data) {
  const { topic, insights, sources, generatedAt, priority } = data;

  let body = '';

  body += `# Research: ${topic}\n\n`;

  if (priority && priority !== 'normal') {
    body += `**Priority:** ${priority.toUpperCase()}\n\n`;
  }

  body += `## Summary\n\n`;
  body += `Research compiled from ${sources.length} sources on ${new Date(generatedAt).toLocaleDateString()}.\n\n`;

  if (insights.keyPoints && insights.keyPoints.length > 0) {
    body += `## Key Findings\n\n`;

    for (const point of insights.keyPoints) {
      body += `- ${point.text}\n`;
      body += `  - Source: [${point.sourceTitle}](${point.source})\n\n`;
    }
  }

  if (insights.topics && insights.topics.length > 0) {
    body += `## Related Topics\n\n`;
    body += insights.topics.map(t => `\`${t}\``).join(' • ');
    body += `\n\n`;
  }

  body += `## Sources\n\n`;

  for (const source of insights.sources) {
    body += `### [${source.title}](${source.url})\n\n`;
    if (source.snippet) {
      body += `${source.snippet}\n\n`;
    }
    body += `*Source: ${source.domain}*\n\n`;
    body += `---\n\n`;
  }

  if (insights.actionable) {
    body += `## Next Steps\n\n`;
    body += `- [ ] Review findings\n`;
    body += `- [ ] Create implementation plan\n`;
    body += `- [ ] Assign to team member\n`;
    body += `- [ ] Set timeline\n\n`;
  }

  body += `---\n\n`;
  body += `*Generated by [web-to-github-issue](https://claudecodeplugins.io) plugin*\n`;
  body += `*Research date: ${new Date(generatedAt).toLocaleString()}*\n`;

  return body;
}

/**
 * Generate issue title from topic and insights
 */
export function generateIssueTitle(topic, insights) {
  if (insights.actionable) {
    return `Implement: ${topic}`;
  }

  if (insights.detectedPriority === 'urgent') {
    return `[URGENT] Research: ${topic}`;
  }

  return `Research: ${topic}`;
}

/**
 * Determine labels based on priority and content
 */
export function determineLabels(baseLabels, priority, insights) {
  const labels = [...baseLabels];

  if (priority === 'urgent' || priority === 'high') {
    labels.push('priority-high');
  }

  if (insights.actionable) {
    labels.push('actionable');
  }

  // Remove duplicates
  return [...new Set(labels)];
}
