# Crypto Portfolio Tracker Plugin

Professional-grade cryptocurrency portfolio tracking and analysis for Claude Code, providing institutional-quality metrics, risk assessment, and optimization recommendations.

## Features

###  Position Tracking
- **Real-time price updates** from multiple exchanges (CoinGecko, Binance, Coinbase)
- **Comprehensive PnL calculations** with entry/exit tracking
- **Multi-exchange support** for accurate pricing
- **Stop-loss and take-profit monitoring**
- **Position history** with complete audit trail
- **Alert system** for significant price movements

###  Portfolio Analysis
- **Risk metrics**: Sharpe ratio, Sortino ratio, maximum drawdown
- **Volatility analysis**: Daily, weekly, monthly, and annual
- **Correlation matrix**: Identify over-concentrated positions
- **Value at Risk (VaR)**: 95% and 99% confidence levels
- **Herfindahl Index**: Measure portfolio concentration
- **Performance attribution**: Understand what's driving returns

###  Rebalancing Engine
- **Optimal allocation** calculations using Modern Portfolio Theory
- **Automated rebalancing** recommendations
- **Tax-aware** trading suggestions
- **Cost estimation** for rebalancing actions
- **Threshold-based** or time-based rebalancing

###  Advanced Analytics
- **Risk-adjusted returns** optimization
- **Diversification scoring**
- **Market regime detection**
- **Drawdown analysis**
- **Monte Carlo simulations** for future scenarios

## Installation

```bash
/plugin install crypto-portfolio-tracker@claude-code-plugins-plus
```

## FREE Data Sources: No Premium APIs Required

**All portfolio tracking uses 100% free data sources** - no CryptoCompare Pro or Messari subscriptions needed.

### Quick Comparison

| Data Source | Paid Alternatives | FREE (This Plugin) |
|-------------|------------------|-------------------|
| **Price Data** | CryptoCompare Pro ($79/mo) | CoinGecko: **$0** |
| **Exchange Data** | Messari Pro ($99/mo) | Binance/Coinbase API: **$0** |
| **Historical Data** | Kaiko ($500/mo) | CoinGecko/Binance: **$0** |
| **Portfolio Analytics** | TradingView Pro ($60/mo) | Built-in: **$0** |

**Annual Savings: $948-7,188** for professional portfolio tracking.

### Free APIs Used by This Plugin

#### 1. CoinGecko (Primary - FREE)

**What:** 10,000+ cryptocurrencies with real-time pricing

**Free Tier:**
- 10-50 calls/minute (generous)
- No API key required
- Full price history
- Market cap, volume, 24h changes

**Setup:**
```json
{
  "priceFeeds": {
    "primary": "coingecko"  // Already configured - FREE
  }
}
```

**Cost:** $0 (no signup required)

**API:** [coingecko.com/api](https://www.coingecko.com/api)

#### 2. Binance API (Fallback - FREE)

**What:** Real-time exchange data from world's largest crypto exchange

**Free Tier:**
- 1,200 requests/minute
- No API key required for public data
- Real-time orderbook data
- 24h ticker statistics

**Setup:**
```json
{
  "priceFeeds": {
    "fallback": ["binance"]  // Already configured - FREE
  }
}
```

**Cost:** $0

**API:** [binance-docs.github.io/apidocs](https://binance-docs.github.io/apidocs)

#### 3. Coinbase API (Fallback - FREE)

**What:** Institutional-grade pricing from regulated US exchange

**Free Tier:**
- 10,000 requests/hour
- No authentication for public endpoints
- OHLCV candles
- Real-time ticker

**Setup:**
```json
{
  "priceFeeds": {
    "fallback": ["coinbase"]  // Already configured - FREE
  }
}
```

**Cost:** $0

**API:** [docs.cloud.coinbase.com](https://docs.cloud.coinbase.com)

### Cost Comparison: Portfolio Tracking

#### Paid Approach (Premium Tools)

**Annual Subscriptions:**
- CryptoCompare Pro: $79/mo → $948/year (historical data)
- Messari Pro: $99/mo → $1,188/year (fundamentals)
- Kaiko: $500/mo → $6,000/year (institutional data)
- TradingView Pro: $60/mo → $720/year (charts/analytics)
- **Total: $8,856/year**

#### Free Approach (This Plugin)

**Annual Subscriptions:**
- CoinGecko: $0
- Binance API: $0
- Coinbase API: $0
- Built-in analytics: $0
- **Total: $0/year**

**Savings: $8,856/year** with identical portfolio tracking quality.

### Real Use Case Example

#### Portfolio Value Calculation

**Paid Approach (CryptoCompare Pro):**
```javascript
// $948/year subscription
const response = await fetch(
  `https://min-api.cryptocompare.com/data/price?fsym=BTC&tsyms=USD&api_key=${CRYPTOCOMPARE_KEY}`
);
```

**Free Approach (CoinGecko - This Plugin):**
```javascript
// $0/year - already configured
const response = await fetch(
  'https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=usd'
);
```

**Cost:** $0 (vs $948/year)
**Data Quality:** Identical (aggregated from same exchanges)

### Data Quality Comparison

| Metric | Paid (CryptoCompare) | FREE (CoinGecko) |
|--------|---------------------|------------------|
| **Cryptocurrencies** | 6,000+ | 10,000+ ✅ |
| **Update Frequency** | Real-time | Real-time ✅ |
| **Historical Data** | Full history | Full history ✅ |
| **Exchange Coverage** | 200+ exchanges | 600+ exchanges ✅ |
| **Cost** | $948/year | $0/year ✅ |

**CoinGecko actually has MORE comprehensive coverage than paid alternatives.**

### Performance Comparison

| Feature | Paid Tools | This Plugin (Free) |
|---------|-----------|-------------------|
| **Real-time Prices** | ✅ | ✅ |
| **Portfolio PnL** | ✅ | ✅ |
| **Risk Metrics** | ✅ | ✅ (Sharpe, Sortino, VaR) |
| **Rebalancing** | ✅ | ✅ (Modern Portfolio Theory) |
| **Alerts** | ✅ | ✅ |
| **Tax Tracking** | ✅ | ✅ (FIFO, LIFO) |
| **Cost** | $8,856/year | $0/year |

**This plugin provides institutional-grade features at zero cost.**

### When Free APIs Are NOT Enough

**Use paid APIs if:**
- You need <100ms latency for high-frequency trading
- You require tick-by-tick orderbook data
- Your firm needs Bloomberg Terminal integration
- You manage $100M+ AUM requiring institutional SLA

**For 99.9% of crypto investors:** Free APIs are more than sufficient.

### Rate Limit Handling

This plugin intelligently manages rate limits:

```json
{
  "priceFeeds": {
    "primary": "coingecko",        // Try CoinGecko first (50 calls/min)
    "fallback": ["binance", "coinbase"],  // Auto-fallback if rate limited
    "updateInterval": 300           // 5 min updates (well under limits)
  }
}
```

**Result:** Never hit rate limits with 300-second update intervals.

### Resources

- **CoinGecko API:** [coingecko.com/api](https://www.coingecko.com/api) (FREE, no key)
- **Binance API:** [binance-docs.github.io](https://binance-docs.github.io/apidocs) (FREE)
- **Coinbase API:** [docs.cloud.coinbase.com](https://docs.cloud.coinbase.com) (FREE)
- **Rate Limits:** All free tiers support 300+ requests/hour (plugin uses ~12/hour)

**Bottom Line:** This plugin already uses 100% free data sources. Save $8,856/year vs premium tools with superior data coverage.

---

## Usage

### Track a New Position

```
/track-position

I'll help you track a crypto position. Please provide:
- Symbol (BTC, ETH, etc.)
- Entry price
- Quantity
- Entry date
- Target price (optional)
- Stop loss (optional)
```

### Analyze Portfolio

```
/portfolio-analysis

Analyzing your complete crypto portfolio...
- Calculating risk metrics
- Evaluating diversification
- Generating rebalancing recommendations
```

### Set Price Alerts

```
/set-alert BTC above 50000
/set-alert ETH below 2000
/set-alert SOL volatility 20%
```

## Configuration

Create a `.crypto-portfolio.json` configuration file in your project root:

```json
{
  "priceFeeds": {
    "primary": "coingecko",
    "fallback": ["binance", "coinbase"],
    "updateInterval": 300
  },
  "alerts": {
    "profitTarget": 0.20,
    "lossWarning": -0.10,
    "criticalLoss": -0.25,
    "volatilitySpike": 0.15
  },
  "rebalancing": {
    "strategy": "threshold",
    "threshold": 0.15,
    "frequency": "monthly",
    "minTradeSize": 100,
    "taxStrategy": "FIFO"
  },
  "riskManagement": {
    "maxPositionSize": 0.30,
    "maxDrawdown": 0.35,
    "stopLossDefault": 0.25
  }
}
```

## Commands

### Position Management

| Command | Description | Shortcut |
|---------|-------------|----------|
| `/track-position` | Track a new crypto position | `tp` |
| `/update-position` | Update an existing position | `up` |
| `/close-position` | Close and record a position | `cp` |
| `/position-history` | View position history | `ph` |

### Analytics

| Command | Description | Shortcut |
|---------|-------------|----------|
| `/portfolio-analysis` | Full portfolio analysis | `pa` |
| `/risk-metrics` | Calculate risk metrics | `rm` |
| `/correlation-matrix` | View correlations | `cm` |
| `/rebalancing-plan` | Get rebalancing suggestions | `rp` |

### Alerts & Monitoring

| Command | Description | Shortcut |
|---------|-------------|----------|
| `/set-alert` | Set price alerts | `sa` |
| `/view-alerts` | View active alerts | `va` |
| `/alert-history` | View triggered alerts | `ah` |

## Database Schema

The plugin uses a structured database to track all portfolio data:

```sql
-- Main positions table
CREATE TABLE crypto_positions (
    id UUID PRIMARY KEY,
    symbol VARCHAR(10) NOT NULL,
    entry_price DECIMAL(20,8) NOT NULL,
    quantity DECIMAL(20,8) NOT NULL,
    entry_date TIMESTAMP NOT NULL,
    current_price DECIMAL(20,8),
    target_price DECIMAL(20,8),
    stop_loss DECIMAL(20,8),
    status VARCHAR(20) DEFAULT 'open',
    exchange VARCHAR(50),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Position history for tracking changes
CREATE TABLE position_history (
    id UUID PRIMARY KEY,
    position_id UUID REFERENCES crypto_positions(id),
    price DECIMAL(20,8) NOT NULL,
    value DECIMAL(20,8) NOT NULL,
    pnl DECIMAL(20,8) NOT NULL,
    pnl_percentage DECIMAL(10,4) NOT NULL,
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Alerts configuration
CREATE TABLE alerts (
    id UUID PRIMARY KEY,
    position_id UUID REFERENCES crypto_positions(id),
    alert_type VARCHAR(50) NOT NULL,
    condition VARCHAR(20) NOT NULL,
    threshold DECIMAL(20,8) NOT NULL,
    status VARCHAR(20) DEFAULT 'active',
    triggered_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

## API Integration

The plugin integrates with multiple cryptocurrency data sources:

### CoinGecko API
- Real-time prices for 10,000+ cryptocurrencies
- Historical price data
- Market cap and volume information

### Binance API
- Real-time trading data
- Order book depth
- 24-hour statistics

### Coinbase API
- Institutional-grade price feeds
- Historical OHLCV data
- Real-time WebSocket streams

## Risk Metrics Explained

### Sharpe Ratio
Measures risk-adjusted returns. Values:
- < 0: Poor (returns below risk-free rate)
- 0-0.5: Suboptimal
- 0.5-1.0: Acceptable
- 1.0-2.0: Good
- > 2.0: Excellent

### Value at Risk (VaR)
Statistical measure of potential loss:
- 95% VaR: Maximum loss expected 95% of the time
- 99% VaR: Maximum loss expected 99% of the time

### Maximum Drawdown
Largest peak-to-trough decline:
- < 10%: Low risk
- 10-20%: Moderate risk
- 20-30%: Elevated risk
- > 30%: High risk

### Herfindahl Index
Measures portfolio concentration:
- < 0.15: Well diversified
- 0.15-0.25: Moderate concentration
- > 0.25: High concentration

## Performance Optimization

### Caching Strategy
- Price data cached for 5 minutes
- Historical data cached for 24 hours
- Correlation matrices cached for 1 hour

### Rate Limiting
- Automatic rate limit handling
- Exponential backoff on API errors
- Fallback to secondary data sources

### Batch Operations
- Batch price updates for efficiency
- Grouped database writes
- Optimized correlation calculations

## Security Considerations

### API Key Management
- Store API keys in environment variables
- Never commit keys to version control
- Use read-only keys where possible

### Data Privacy
- Local database storage by default
- Optional encryption for sensitive data
- No third-party data sharing

### Input Validation
- Sanitize all user inputs
- Validate cryptocurrency symbols
- Check numerical bounds

## Troubleshooting

### Common Issues

**Price data not updating**
- Check API key configuration
- Verify internet connectivity
- Check rate limits

**Incorrect PnL calculations**
- Verify entry prices and dates
- Check for stock splits or airdrops
- Ensure correct quantity tracking

**High correlation warnings**
- Review asset selection
- Consider adding uncorrelated assets
- Check time period for correlation calculation

## Advanced Features

### Tax Reporting
- FIFO/LIFO cost basis tracking
- Capital gains calculations
- Export to TurboTax/CoinTracker format

### Backtesting
- Test strategies on historical data
- Monte Carlo simulations
- Walk-forward analysis

### Automation
- Automated rebalancing execution
- Stop-loss order placement
- DCA (Dollar Cost Averaging) scheduling

## Performance Benchmarks

- Position tracking: < 100ms
- Portfolio analysis: < 2 seconds for 50 positions
- Correlation calculation: < 500ms for 20 assets
- Rebalancing optimization: < 1 second

## Contributing

This plugin is part of the Claude Code Plugins marketplace. To contribute:

1. Fork the repository
2. Create a feature branch
3. Add tests for new features
4. Submit a pull request

## License

MIT License - See LICENSE file for details

## Support

- GitHub Issues: [Report bugs](https://github.com/jeremylongshore/claude-code-plugins/issues)
- Discord: Join the Claude Code community
- Documentation: [Full API docs](https://docs.claude-code-plugins.com)

## Changelog

### v1.0.0 (2024-10-11)
- Initial release
- Position tracking with real-time updates
- Portfolio analysis with risk metrics
- Rebalancing recommendations
- Alert system
- Multi-exchange price feeds

---

*Built with ️ for crypto investors by Intent Solutions IO*