---
name: juicebox-prod-checklist
description: |
  Execute Juicebox production deployment checklist.
  Use when preparing for production launch, validating deployment readiness,
  or performing pre-launch reviews.
  Trigger with phrases like "juicebox production", "deploy juicebox prod",
  "juicebox launch checklist", "juicebox go-live".
allowed-tools: Read, Grep, Bash(curl:*)
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# Juicebox Production Checklist

## Overview
Complete production readiness checklist for Juicebox integration deployment.

## Prerequisites
- Development and staging testing complete
- Production environment provisioned
- Monitoring infrastructure ready

## Production Readiness Checklist

### 1. API Configuration
```markdown
- [ ] Production API key obtained and configured
- [ ] API key stored in secret manager (not env vars)
- [ ] Key rotation schedule documented
- [ ] Backup API key configured
- [ ] Rate limits understood and within quota
```

### 2. Error Handling
```markdown
- [ ] All error codes handled gracefully
- [ ] Retry logic with exponential backoff
- [ ] Circuit breaker pattern implemented
- [ ] Fallback behavior defined
- [ ] Error logging and alerting configured
```

### 3. Performance
```markdown
- [ ] Response time SLAs defined
- [ ] Caching layer implemented
- [ ] Connection pooling configured
- [ ] Timeout values set appropriately
- [ ] Load testing completed
```

### 4. Security
```markdown
- [ ] API key not exposed in client-side code
- [ ] HTTPS enforced for all communications
- [ ] Audit logging enabled
- [ ] Access controls implemented
- [ ] PII handling compliant with regulations
```

### 5. Monitoring
```markdown
- [ ] Health check endpoint configured
- [ ] Metrics collection enabled
- [ ] Alerting rules defined
- [ ] Dashboard created
- [ ] On-call runbook documented
```

### 6. Documentation
```markdown
- [ ] Integration architecture documented
- [ ] API usage documented for team
- [ ] Troubleshooting guide created
- [ ] Escalation path defined
- [ ] Support contact information recorded
```

## Validation Scripts

### API Connectivity Check
```bash
#!/bin/bash
# validate-juicebox-prod.sh

echo "=== Juicebox Production Validation ==="

# Check API key is set
if [ -z "$JUICEBOX_API_KEY" ]; then
  echo "FAIL: JUICEBOX_API_KEY not set"
  exit 1
fi

# Test health endpoint
HEALTH=$(curl -s -w "%{http_code}" -o /dev/null https://api.juicebox.ai/v1/health)
if [ "$HEALTH" != "200" ]; then
  echo "FAIL: Health check returned $HEALTH"
  exit 1
fi
echo "PASS: Health check"

# Test authentication
AUTH=$(curl -s -w "%{http_code}" -o /dev/null \
  -H "Authorization: Bearer $JUICEBOX_API_KEY" \
  https://api.juicebox.ai/v1/auth/me)
if [ "$AUTH" != "200" ]; then
  echo "FAIL: Auth check returned $AUTH"
  exit 1
fi
echo "PASS: Authentication"

# Test sample search
SEARCH=$(curl -s -w "%{http_code}" -o /dev/null \
  -H "Authorization: Bearer $JUICEBOX_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"query":"test","limit":1}' \
  https://api.juicebox.ai/v1/search)
if [ "$SEARCH" != "200" ]; then
  echo "FAIL: Search test returned $SEARCH"
  exit 1
fi
echo "PASS: Search functionality"

echo "=== All production checks passed ==="
```

### Integration Test Suite
```typescript
// tests/production-readiness.test.ts
import { describe, it, expect } from 'vitest';
import { JuiceboxClient } from '@juicebox/sdk';

describe('Production Readiness', () => {
  const client = new JuiceboxClient({
    apiKey: process.env.JUICEBOX_API_KEY!
  });

  it('authenticates successfully', async () => {
    const user = await client.auth.me();
    expect(user.id).toBeDefined();
  });

  it('performs search within SLA', async () => {
    const start = Date.now();
    const results = await client.search.people({
      query: 'software engineer',
      limit: 10
    });
    const duration = Date.now() - start;

    expect(results.profiles.length).toBeGreaterThan(0);
    expect(duration).toBeLessThan(5000); // 5s SLA
  });

  it('handles rate limiting gracefully', async () => {
    // Implementation depends on your retry logic
  });
});
```

## Go-Live Checklist

```markdown
## Day-of-Launch Checklist

### Pre-Launch (T-1 hour)
- [ ] All validation scripts pass
- [ ] Monitoring dashboards open
- [ ] On-call team notified
- [ ] Rollback plan reviewed

### Launch
- [ ] Feature flag enabled
- [ ] Traffic gradually increased
- [ ] Error rates monitored
- [ ] Performance metrics checked

### Post-Launch (T+1 hour)
- [ ] All systems nominal
- [ ] No unexpected errors
- [ ] Customer feedback monitored
- [ ] Success metrics tracked
```

## Resources
- [Production Best Practices](https://juicebox.ai/docs/production)
- [Status Page](https://status.juicebox.ai)

## Next Steps
After production launch, see `juicebox-upgrade-migration` for SDK updates.
