---
name: lindy-migration-deep-dive
description: |
  Advanced migration strategies for Lindy AI integrations.
  Use when migrating from other platforms, consolidating agents,
  or performing major architecture changes.
  Trigger with phrases like "lindy migration", "migrate to lindy",
  "lindy platform migration", "switch to lindy".
allowed-tools: Read, Write, Edit, Bash(node:*)
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# Lindy Migration Deep Dive

## Overview
Advanced migration strategies for moving to or upgrading Lindy AI integrations.

## Prerequisites
- Source platform documentation
- Target Lindy environment ready
- Migration timeline approved
- Rollback plan defined

## Migration Scenarios

### Scenario 1: From Custom AI to Lindy

**Assessment Phase:**
```typescript
// migration/assess.ts
interface MigrationAssessment {
  sourceAgents: number;
  sourceWorkflows: number;
  complexity: 'simple' | 'moderate' | 'complex';
  estimatedDuration: string;
  risks: string[];
}

async function assessMigration(source: any): Promise<MigrationAssessment> {
  // Analyze existing system
  const agents = await source.getAgents();
  const workflows = await source.getWorkflows();

  const complexity = agents.length > 10 || workflows.length > 5
    ? 'complex'
    : agents.length > 3
    ? 'moderate'
    : 'simple';

  return {
    sourceAgents: agents.length,
    sourceWorkflows: workflows.length,
    complexity,
    estimatedDuration: complexity === 'complex' ? '2-4 weeks' : '1 week',
    risks: [
      'Feature parity gaps',
      'Data format differences',
      'Integration rewiring',
    ],
  };
}
```

### Scenario 2: Agent Consolidation

**Before:**
```
┌─────────────┐ ┌─────────────┐ ┌─────────────┐
│  Agent A    │ │  Agent B    │ │  Agent C    │
│  (Support)  │ │  (Support)  │ │  (Support)  │
└─────────────┘ └─────────────┘ └─────────────┘
      │               │               │
      └───────────────┴───────────────┘
                      │
              (Duplicated logic)
```

**After:**
```
┌─────────────────────────────────────────────┐
│           Unified Support Agent             │
│  (Consolidated logic, shared context)       │
└─────────────────────────────────────────────┘
```

```typescript
// migration/consolidate.ts
async function consolidateAgents(agentIds: string[]) {
  const lindy = new Lindy({ apiKey: process.env.LINDY_API_KEY });

  // Collect all instructions
  const agents = await Promise.all(
    agentIds.map(id => lindy.agents.get(id))
  );

  // Merge instructions
  const mergedInstructions = agents
    .map(a => `## ${a.name}\n${a.instructions}`)
    .join('\n\n');

  // Collect all tools
  const allTools = [...new Set(agents.flatMap(a => a.tools))];

  // Create consolidated agent
  const consolidated = await lindy.agents.create({
    name: 'Unified Support Agent',
    instructions: `
      You are a unified support agent combining multiple specializations.

      ${mergedInstructions}

      Use the appropriate section based on the user's query.
    `,
    tools: allTools,
  });

  console.log(`Consolidated ${agents.length} agents into: ${consolidated.id}`);

  return consolidated;
}
```

### Scenario 3: Multi-Environment Migration

```typescript
// migration/multi-env.ts
interface MigrationPlan {
  phases: MigrationPhase[];
  rollbackCheckpoints: string[];
}

interface MigrationPhase {
  name: string;
  environment: 'development' | 'staging' | 'production';
  steps: string[];
  duration: string;
  successCriteria: string[];
}

const migrationPlan: MigrationPlan = {
  phases: [
    {
      name: 'Development Migration',
      environment: 'development',
      steps: [
        'Export agents from source',
        'Transform to Lindy format',
        'Import to Lindy dev',
        'Run integration tests',
        'Fix any issues',
      ],
      duration: '1 week',
      successCriteria: [
        'All agents imported',
        'Integration tests passing',
        'No critical errors in logs',
      ],
    },
    {
      name: 'Staging Migration',
      environment: 'staging',
      steps: [
        'Deploy to staging',
        'Run load tests',
        'Parallel run with source',
        'Compare outputs',
        'Fix discrepancies',
      ],
      duration: '1 week',
      successCriteria: [
        'Load tests passing',
        'Output parity > 95%',
        'Latency within SLA',
      ],
    },
    {
      name: 'Production Migration',
      environment: 'production',
      steps: [
        'Deploy to production (canary)',
        'Gradually shift traffic',
        'Monitor metrics',
        'Complete cutover',
        'Decommission source',
      ],
      duration: '2 weeks',
      successCriteria: [
        'No increase in errors',
        'Latency within SLA',
        'User satisfaction maintained',
      ],
    },
  ],
  rollbackCheckpoints: [
    'After dev import',
    'After staging deployment',
    'After 25% traffic shift',
    'After 50% traffic shift',
  ],
};
```

### Data Migration

```typescript
// migration/data.ts
interface DataMigration {
  source: string;
  destination: string;
  transform: (data: any) => any;
}

async function migrateData(config: DataMigration) {
  const lindy = new Lindy({ apiKey: process.env.LINDY_API_KEY });

  // Export from source
  console.log('Exporting from source...');
  const sourceData = await exportFromSource(config.source);

  // Transform data
  console.log('Transforming data...');
  const transformedData = sourceData.map(config.transform);

  // Validate transformed data
  console.log('Validating...');
  const validationErrors = validateData(transformedData);
  if (validationErrors.length > 0) {
    throw new Error(`Validation failed: ${validationErrors.join(', ')}`);
  }

  // Import to Lindy
  console.log('Importing to Lindy...');
  for (const item of transformedData) {
    await lindy.agents.create(item);
  }

  console.log(`Migrated ${transformedData.length} items`);
}

// Transform functions for different sources
const transforms = {
  openai: (agent: any) => ({
    name: agent.name,
    instructions: agent.instructions,
    tools: mapOpenAITools(agent.tools),
  }),

  langchain: (agent: any) => ({
    name: agent.name,
    instructions: agent.prompt_template,
    tools: mapLangChainTools(agent.tools),
  }),

  custom: (agent: any) => ({
    name: agent.title,
    instructions: agent.system_prompt,
    tools: agent.enabled_tools || [],
  }),
};
```

### Rollback Procedures

```typescript
// migration/rollback.ts
interface RollbackState {
  checkpoint: string;
  timestamp: Date;
  agentSnapshots: Map<string, any>;
  automationSnapshots: Map<string, any>;
}

class RollbackManager {
  private states: RollbackState[] = [];
  private lindy: Lindy;

  constructor() {
    this.lindy = new Lindy({ apiKey: process.env.LINDY_API_KEY });
  }

  async createCheckpoint(name: string): Promise<void> {
    console.log(`Creating checkpoint: ${name}`);

    const agents = await this.lindy.agents.list();
    const automations = await this.lindy.automations.list();

    const state: RollbackState = {
      checkpoint: name,
      timestamp: new Date(),
      agentSnapshots: new Map(agents.map(a => [a.id, a])),
      automationSnapshots: new Map(automations.map(a => [a.id, a])),
    };

    this.states.push(state);
    console.log(`Checkpoint created with ${agents.length} agents`);
  }

  async rollback(checkpointName: string): Promise<void> {
    const state = this.states.find(s => s.checkpoint === checkpointName);
    if (!state) {
      throw new Error(`Checkpoint not found: ${checkpointName}`);
    }

    console.log(`Rolling back to: ${checkpointName}`);

    // Delete new agents
    const currentAgents = await this.lindy.agents.list();
    for (const agent of currentAgents) {
      if (!state.agentSnapshots.has(agent.id)) {
        await this.lindy.agents.delete(agent.id);
      }
    }

    // Restore modified agents
    for (const [id, snapshot] of state.agentSnapshots) {
      await this.lindy.agents.update(id, snapshot);
    }

    console.log(`Rollback to ${checkpointName} complete`);
  }
}
```

## Migration Checklist
```markdown
[ ] Source system documented
[ ] Migration plan approved
[ ] Rollback procedures tested
[ ] Data transformation validated
[ ] Feature parity confirmed
[ ] Integration tests created
[ ] Load tests passed
[ ] Parallel run completed
[ ] Cutover window scheduled
[ ] Monitoring enhanced
[ ] Support team briefed
```

## Output
- Migration assessment
- Consolidation strategy
- Multi-environment plan
- Data transformation
- Rollback procedures

## Error Handling
| Issue | Cause | Solution |
|-------|-------|----------|
| Data loss | Transform error | Validate before import |
| Parity gap | Feature difference | Document and workaround |
| Rollback fail | Incomplete checkpoint | Create full snapshots |

## Examples

### Complete Migration Script
```bash
#!/bin/bash
# migrate-to-lindy.sh

echo "Starting Lindy migration..."

# Phase 1: Assessment
npm run migration:assess

# Phase 2: Export
npm run migration:export

# Phase 3: Transform
npm run migration:transform

# Phase 4: Validate
npm run migration:validate

# Phase 5: Import (with checkpoint)
npm run migration:checkpoint create pre-import
npm run migration:import

# Phase 6: Test
npm run migration:test

echo "Migration complete!"
```

## Resources
- [Lindy Migration Guide](https://docs.lindy.ai/migration)
- [Data Import API](https://docs.lindy.ai/api/import)
- [Best Practices](https://docs.lindy.ai/migration/best-practices)

## Next Steps
This completes the Flagship tier skills. Consider reviewing Standard and Pro skills for comprehensive coverage.
