---
name: lindy-prod-checklist
description: |
  Production readiness checklist for Lindy AI deployments.
  Use when preparing for production, reviewing deployment,
  or auditing production setup.
  Trigger with phrases like "lindy production", "lindy prod ready",
  "lindy go live", "lindy deployment checklist".
allowed-tools: Read, Write, Edit
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# Lindy Prod Checklist

## Overview
Comprehensive production readiness checklist for Lindy AI deployments.

## Prerequisites
- Completed development and testing
- Production Lindy account
- Deployment infrastructure ready

## Production Checklist

### Authentication & Security
```markdown
[ ] Production API key generated
[ ] API key stored in secret manager (not env file)
[ ] Key rotation process documented
[ ] Different keys for each environment
[ ] Keys have appropriate scopes/permissions
[ ] Service accounts configured (not personal keys)
```

### Agent Configuration
```markdown
[ ] All agents tested with production-like data
[ ] Agent instructions reviewed and finalized
[ ] Tool permissions minimized (least privilege)
[ ] Timeout values appropriate for workloads
[ ] Error handling tested for all failure modes
[ ] Fallback behaviors defined
```

### Monitoring & Observability
```markdown
[ ] Logging configured and tested
[ ] Error alerting set up (PagerDuty/Slack/etc)
[ ] Usage metrics dashboards created
[ ] Rate limit alerts configured
[ ] Latency monitoring enabled
[ ] Cost tracking implemented
```

### Performance & Reliability
```markdown
[ ] Load testing completed
[ ] Rate limit handling implemented
[ ] Retry logic with exponential backoff
[ ] Circuit breaker pattern for failures
[ ] Graceful degradation defined
[ ] SLA targets documented
```

### Compliance & Documentation
```markdown
[ ] Data handling documented
[ ] Privacy review completed
[ ] Security review completed
[ ] Runbooks created for incidents
[ ] Escalation paths defined
[ ] On-call schedule set up
```

## Implementation

### Health Check Endpoint
```typescript
// health/lindy.ts
import { Lindy } from '@lindy-ai/sdk';

export async function checkLindyHealth(): Promise<HealthStatus> {
  const lindy = new Lindy({ apiKey: process.env.LINDY_API_KEY });
  const start = Date.now();

  try {
    await lindy.users.me();
    const latency = Date.now() - start;

    return {
      status: latency < 1000 ? 'healthy' : 'degraded',
      latency,
      timestamp: new Date().toISOString(),
    };
  } catch (error: any) {
    return {
      status: 'unhealthy',
      error: error.message,
      timestamp: new Date().toISOString(),
    };
  }
}
```

### Pre-Deployment Validation
```typescript
async function preDeploymentCheck(): Promise<boolean> {
  const checks = {
    apiKey: !!process.env.LINDY_API_KEY,
    environment: process.env.LINDY_ENVIRONMENT === 'production',
    connectivity: false,
    agents: false,
  };

  const lindy = new Lindy({ apiKey: process.env.LINDY_API_KEY });

  try {
    await lindy.users.me();
    checks.connectivity = true;

    const agents = await lindy.agents.list();
    checks.agents = agents.length > 0;
  } catch (e) {
    // Failed checks
  }

  const passed = Object.values(checks).every(Boolean);
  console.log('Pre-deployment checks:', checks);
  console.log(`Status: ${passed ? 'PASSED' : 'FAILED'}`);

  return passed;
}
```

## Output
- Complete production checklist
- Health check implementation
- Pre-deployment validation script
- Go/no-go criteria defined

## Error Handling
| Check | Failure Action | Severity |
|-------|----------------|----------|
| API Key | Block deploy | Critical |
| Connectivity | Retry/alert | High |
| Agents exist | Warning | Medium |
| Monitoring | Document gap | Medium |

## Examples

### Deployment Gate Script
```bash
#!/bin/bash
# deploy-gate.sh

echo "Running Lindy pre-deployment checks..."

npx ts-node scripts/pre-deployment-check.ts
if [ $? -ne 0 ]; then
  echo "Pre-deployment checks FAILED"
  exit 1
fi

echo "All checks passed. Proceeding with deployment."
```

## Resources
- [Lindy Production Guide](https://docs.lindy.ai/production)
- [SLA Information](https://lindy.ai/sla)
- [Support](https://support.lindy.ai)

## Next Steps
Proceed to `lindy-upgrade-migration` for version upgrades.
