# AI/ML Engineering Pack - Installation Guide

Complete installation guide for the AI/ML Engineering Pack, a professional toolkit for prompt engineering, LLM integration, RAG systems, and AI safety.

## Prerequisites

### Required Software
- **Claude Code CLI** (version 0.1.0 or higher)
- **Python 3.10+** (for running generated code examples)
- **Node.js 18+** (for TypeScript examples and API scaffolding)
- **Git** (for cloning examples and templates)

### Optional Software (Recommended)
- **Docker 20.10+** (for containerized deployments)
- **Poetry or pip** (Python package management)
- **npm or yarn** (Node.js package management)

### API Keys (As Needed)
This pack generates code for multiple LLM providers. You'll need API keys based on your chosen provider:
- **OpenAI API key** (gpt-4-turbo, gpt-3.5-turbo)
- **Anthropic API key** (claude-3-opus, claude-3-sonnet, claude-3-haiku)
- **Google Cloud credentials** (gemini-pro, gemini-flash)
- **Cohere API key** (for reranking in RAG systems)

Vector database credentials (if using hosted solutions):
- **Pinecone API key**
- **Qdrant Cloud credentials**
- **Weaviate Cloud credentials**

## Installation Steps

### 1. Install Claude Code CLI

If you haven't already installed Claude Code:

```bash
# macOS/Linux
curl -fsSL https://claude.ai/install.sh | sh

# Verify installation
claude --version
```

### 2. Add Plugin Marketplace

Add the claude-code-plugins-plus marketplace (if not already added):

```bash
claude plugin marketplace add jeremylongshore/claude-code-plugins
```

### 3. Install AI/ML Engineering Pack

Install the complete pack with all 12 plugins:

```bash
claude plugin install ai-ml-engineering-pack@claude-code-plugins-plus
```

This installs:
- **Prompt Engineering** (3 plugins)
- **LLM Integration** (3 plugins)
- **RAG Systems** (3 plugins)
- **AI Safety** (3 plugins)

### 4. Verify Installation

Check that all plugins are installed:

```bash
claude plugin list
```

You should see:
```
 prompt-architect (agent)
 prompt-optimizer (agent)
 prompt-template-gen (command - /ptg)
 llm-integration-expert (agent)
 model-selector (agent)
 llm-api-scaffold (command - /las)
 rag-architect (agent)
 vector-db-expert (agent)
 rag-pipeline-gen (command - /rpg)
 ai-safety-expert (agent)
 prompt-injection-defender (agent)
 ai-monitoring-setup (command - /ams)
```

### 5. Test Installation

Test a quick command to verify everything works:

```bash
# Start Claude Code
claude

# Inside Claude Code, test prompt template generator
/ptg
```

You should see Claude generate a prompt template based on your requirements.

## Python Environment Setup

For running generated Python code (RAG pipelines, LLM integrations):

```bash
# Create virtual environment
python3 -m venv venv-aiml
source venv-aiml/bin/activate  # On Windows: venv-aiml\Scripts\activate

# Install common dependencies
pip install openai anthropic google-generativeai
pip install langchain langchain-openai langchain-anthropic
pip install pinecone-client qdrant-client weaviate-client
pip install presidio-analyzer presidio-anonymizer
pip install fastapi uvicorn pydantic
pip install prometheus-client python-dotenv
```

## Node.js Environment Setup

For running generated TypeScript code:

```bash
# Initialize Node.js project
npm init -y

# Install common dependencies
npm install openai @anthropic-ai/sdk @google/generative-ai
npm install langchain @langchain/openai @langchain/anthropic
npm install @pinecone-database/pinecone qdrant-client weaviate-ts-client
npm install fastify zod dotenv
npm install prom-client winston
```

## Docker Setup (Optional)

For containerized deployments generated by `/las` and `/rpg`:

```bash
# Verify Docker is installed
docker --version
docker-compose --version

# Test Docker
docker run hello-world
```

## Configuration

### Environment Variables

Create a `.env` file for your projects (generated code will use this):

```bash
# LLM Provider API Keys
OPENAI_API_KEY=sk-...
ANTHROPIC_API_KEY=sk-ant-...
GOOGLE_API_KEY=...

# Vector Database Credentials
PINECONE_API_KEY=...
PINECONE_ENVIRONMENT=us-west1-gcp
QDRANT_URL=https://your-cluster.qdrant.io
QDRANT_API_KEY=...
WEAVIATE_URL=https://your-cluster.weaviate.network
WEAVIATE_API_KEY=...

# Monitoring
PROMETHEUS_PORT=9090
GRAFANA_PORT=3000

# Cost Control
MONTHLY_BUDGET_USD=1000
COST_ALERT_THRESHOLD=0.8

# Safety Settings
ENABLE_PII_DETECTION=true
ENABLE_TOXICITY_FILTER=true
ENABLE_PROMPT_INJECTION_DEFENSE=true
```

### Cost Limits (Recommended)

Set up cost limits to prevent unexpected charges:

```bash
# OpenAI - Set usage limits in dashboard
# https://platform.openai.com/account/billing/limits

# Anthropic - Configure usage alerts
# https://console.anthropic.com/settings/usage

# GCP - Set budget alerts
# https://console.cloud.google.com/billing/budgets
```

## Verification Checklist

Before starting development, verify:

- [ ] Claude Code CLI installed and working
- [ ] AI/ML Engineering Pack installed (12 plugins)
- [ ] Python 3.10+ environment set up
- [ ] Node.js 18+ environment set up (optional)
- [ ] At least one LLM provider API key configured
- [ ] Vector database credentials (for RAG work)
- [ ] `.env` file created with API keys
- [ ] Docker installed (if using containerized deployments)
- [ ] Cost limits configured on provider dashboards

## Quick Test

Run this quick test to ensure everything is working:

```bash
# Start Claude Code
claude

# Test each category:

# 1. Prompt Engineering
"Help me optimize this prompt: 'Write a summary of the following text...'"
# Claude will use prompt-optimizer agent

# 2. LLM Integration
/las
# Should start generating LLM API integration code

# 3. RAG Systems
/rpg
# Should start generating RAG pipeline

# 4. AI Safety
"Implement PII detection for user inputs"
# Claude will use ai-safety-expert agent
```

## Updating

To update to the latest version:

```bash
# Check for updates
claude plugin update ai-ml-engineering-pack

# Or reinstall
claude plugin uninstall ai-ml-engineering-pack
claude plugin install ai-ml-engineering-pack@claude-code-plugins-plus
```

## Troubleshooting

See [TROUBLESHOOTING.md](./000-docs/157-DR-FAQS-troubleshooting.md) for common issues and solutions.

## Next Steps

Once installed, proceed to:
1. [Quick Start Guide](./QUICK_START.md) - 10-minute walkthrough
2. [Use Cases](./USE_CASES.md) - Real-world examples with ROI
3. [Plugin Reference](./README.md) - Complete plugin documentation

## Support

- **Issues:** https://github.com/jeremylongshore/claude-code-plugins/issues
- **Discussions:** https://github.com/jeremylongshore/claude-code-plugins/discussions
- **Email:** [email protected]

---

**Installation Complete!** Proceed to [QUICK_START.md](./QUICK_START.md) to build your first AI/ML feature.
