---
name: lokalise-ci-integration
description: |
  Configure Lokalise CI/CD integration with GitHub Actions and automated sync.
  Use when setting up automated translation sync, configuring CI pipelines,
  or integrating Lokalise into your build process.
  Trigger with phrases like "lokalise CI", "lokalise GitHub Actions",
  "lokalise automated sync", "CI lokalise", "lokalise pipeline".
allowed-tools: Read, Write, Edit, Bash(gh:*)
version: 1.0.0
license: MIT
author: Jeremy Longshore <jeremy@intentsolutions.io>
---

# Lokalise CI Integration

## Overview
Set up CI/CD pipelines for Lokalise integrations with automated translation sync.

## Prerequisites
- GitHub repository with Actions enabled
- Lokalise API token
- npm/pnpm project configured

## Instructions

### Step 1: Create GitHub Actions Workflow
Create `.github/workflows/lokalise-sync.yml`:

```yaml
name: Lokalise Translation Sync

on:
  push:
    branches: [main]
    paths:
      - 'src/locales/en.json'  # Trigger on source changes
  pull_request:
    branches: [main]
  schedule:
    - cron: '0 6 * * *'  # Daily sync at 6 AM UTC
  workflow_dispatch:  # Manual trigger

env:
  LOKALISE_API_TOKEN: ${{ secrets.LOKALISE_API_TOKEN }}
  LOKALISE_PROJECT_ID: ${{ secrets.LOKALISE_PROJECT_ID }}

jobs:
  sync-translations:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Setup Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'
          cache: 'npm'

      - name: Install Lokalise CLI
        run: |
          curl -sL https://github.com/lokalise/lokalise-cli-2-go/releases/latest/download/lokalise2_linux_x86_64.tar.gz | tar xz
          sudo mv lokalise2 /usr/local/bin/

      - name: Download translations
        run: |
          lokalise2 file download \
            --token "$LOKALISE_API_TOKEN" \
            --project-id "$LOKALISE_PROJECT_ID" \
            --format json \
            --original-filenames=false \
            --bundle-structure "src/locales/%LANG_ISO%.json" \
            --export-empty-as skip \
            --unzip-to .

      - name: Check for changes
        id: changes
        run: |
          if git diff --quiet src/locales/; then
            echo "changed=false" >> $GITHUB_OUTPUT
          else
            echo "changed=true" >> $GITHUB_OUTPUT
          fi

      - name: Commit and push
        if: steps.changes.outputs.changed == 'true' && github.event_name != 'pull_request'
        run: |
          git config user.name "github-actions[bot]"
          git config user.email "github-actions[bot]@users.noreply.github.com"
          git add src/locales/
          git commit -m "chore: sync translations from Lokalise"
          git push
```

### Step 2: Configure Secrets
```bash
# Using GitHub CLI
gh secret set LOKALISE_API_TOKEN --body "your-api-token"
gh secret set LOKALISE_PROJECT_ID --body "123456789.abcdef"

# Or via GitHub UI:
# Settings -> Secrets and variables -> Actions -> New repository secret
```

### Step 3: Upload Source Strings Workflow
Create `.github/workflows/lokalise-upload.yml`:

```yaml
name: Upload Source Strings to Lokalise

on:
  push:
    branches: [main]
    paths:
      - 'src/locales/en.json'

jobs:
  upload:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Install Lokalise CLI
        run: |
          curl -sL https://github.com/lokalise/lokalise-cli-2-go/releases/latest/download/lokalise2_linux_x86_64.tar.gz | tar xz
          sudo mv lokalise2 /usr/local/bin/

      - name: Upload source strings
        env:
          LOKALISE_API_TOKEN: ${{ secrets.LOKALISE_API_TOKEN }}
          LOKALISE_PROJECT_ID: ${{ secrets.LOKALISE_PROJECT_ID }}
        run: |
          lokalise2 file upload \
            --token "$LOKALISE_API_TOKEN" \
            --project-id "$LOKALISE_PROJECT_ID" \
            --file "src/locales/en.json" \
            --lang-iso en \
            --replace-modified \
            --convert-placeholders \
            --detect-icu-plurals \
            --tag-inserted-keys \
            --tags "ci-upload" \
            --poll \
            --poll-timeout 120s
```

### Step 4: PR Preview with Translation Status
```yaml
name: Translation Status Check

on:
  pull_request:
    branches: [main]

jobs:
  check-translations:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Setup Node.js
        uses: actions/setup-node@v4
        with:
          node-version: '20'

      - name: Install dependencies
        run: npm ci

      - name: Check translation coverage
        run: |
          node scripts/check-translations.js
        env:
          LOKALISE_API_TOKEN: ${{ secrets.LOKALISE_API_TOKEN }}
          LOKALISE_PROJECT_ID: ${{ secrets.LOKALISE_PROJECT_ID }}

      - name: Comment on PR
        if: always()
        uses: actions/github-script@v7
        with:
          script: |
            const fs = require('fs');
            const report = fs.readFileSync('translation-report.md', 'utf8');
            github.rest.issues.createComment({
              issue_number: context.issue.number,
              owner: context.repo.owner,
              repo: context.repo.repo,
              body: report
            });
```

## Output
- Automated translation sync pipeline
- PR checks for translation status
- Daily sync scheduled
- Upload on source file changes

## Error Handling
| Issue | Cause | Solution |
|-------|-------|----------|
| Secret not found | Missing configuration | Add secret via `gh secret set` |
| Upload timeout | Large file | Increase poll-timeout |
| Auth failures | Invalid token | Check secret value |
| Rate limited | Too many requests | Add delay or reduce frequency |

## Examples

### Translation Coverage Check Script
```javascript
// scripts/check-translations.js
import { LokaliseApi } from "@lokalise/node-api";
import fs from "fs";

const client = new LokaliseApi({
  apiKey: process.env.LOKALISE_API_TOKEN,
});

async function checkCoverage() {
  const projectId = process.env.LOKALISE_PROJECT_ID;

  // Get project statistics
  const project = await client.projects().get(projectId);

  // Get language statistics
  const languages = await client.languages().list({
    project_id: projectId,
  });

  let report = "## Translation Status Report\n\n";
  report += `**Project:** ${project.name}\n`;
  report += `**Total Keys:** ${project.statistics.keys_total}\n\n`;

  report += "| Language | Progress | Words |\n";
  report += "|----------|----------|-------|\n";

  for (const lang of languages.items) {
    const progress = lang.statistics?.progress ?? 0;
    report += `| ${lang.lang_name} (${lang.lang_iso}) | ${progress}% | ${lang.statistics?.words_total ?? 0} |\n`;
  }

  fs.writeFileSync("translation-report.md", report);
  console.log(report);

  // Fail if critical languages below threshold
  const criticalLanguages = ["es", "fr", "de"];
  for (const lang of languages.items) {
    if (criticalLanguages.includes(lang.lang_iso)) {
      if ((lang.statistics?.progress ?? 0) < 90) {
        console.error(`WARNING: ${lang.lang_iso} is below 90% coverage`);
        process.exit(1);
      }
    }
  }
}

checkCoverage().catch(console.error);
```

### Branch-Based Workflow
```yaml
name: Branch Translation Sync

on:
  push:
    branches:
      - 'feature/*'
      - 'release/*'

jobs:
  sync:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Determine Lokalise branch
        id: branch
        run: |
          # Map Git branch to Lokalise branch
          LOKALISE_BRANCH="${GITHUB_REF_NAME//\//-}"
          echo "lokalise_branch=$LOKALISE_BRANCH" >> $GITHUB_OUTPUT

      - name: Sync with Lokalise branch
        run: |
          lokalise2 file download \
            --token "$LOKALISE_API_TOKEN" \
            --project-id "$LOKALISE_PROJECT_ID:${{ steps.branch.outputs.lokalise_branch }}" \
            --format json \
            --unzip-to ./src/locales
```

### GitLab CI Configuration
```yaml
# .gitlab-ci.yml
lokalise-sync:
  stage: build
  image: node:20
  before_script:
    - curl -sL https://github.com/lokalise/lokalise-cli-2-go/releases/latest/download/lokalise2_linux_x86_64.tar.gz | tar xz
    - mv lokalise2 /usr/local/bin/
  script:
    - lokalise2 file download
        --token "$LOKALISE_API_TOKEN"
        --project-id "$LOKALISE_PROJECT_ID"
        --format json
        --unzip-to ./src/locales
  artifacts:
    paths:
      - src/locales/
  only:
    - main
```

## Resources
- [GitHub Actions Documentation](https://docs.github.com/en/actions)
- [Lokalise CLI Documentation](https://docs.lokalise.com/en/articles/3401683-lokalise-cli-v2)
- [Lokalise GitHub Integration](https://docs.lokalise.com/en/articles/1558810-github)

## Next Steps
For deployment patterns, see `lokalise-deploy-integration`.
